# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeodbIO
                                 A QGIS plugin
 Connects to spatial data from geodb.io via api
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-02-05
        git sha              : $Format:%H$
        copyright            : (C) 2025 by geodb.io
        email                : admin@geodb.io
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QGridLayout, QMessageBox, QFileDialog, QProgressDialog, QWidget
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkReply, QNetworkAccessManager
from qgis.core import (
    QgsVectorLayer,
    QgsVectorFileWriter,
    QgsProject,
    QgsField,
    QgsFields,
    QgsVectorLayerUtils,
    QgsFeature, 
    QgsGeometry,
    QgsCoordinateTransformContext,
    QgsApplication, 
    QgsAuthMethodConfig, 
    QgsProject, 
    QgsNetworkAccessManager,
    QgsEditorWidgetSetup,
    NULL,
    QgsJsonExporter,
)
from qgis.PyQt.QtCore import (
    QSettings, 
    QTranslator, 
    QCoreApplication, 
    QEventLoop, 
    QUrl, 
    QByteArray, 
    QVariant, 
    QMetaType, 
    Qt, 
    QDate, 
    QJsonDocument, 
    QVariant, 
    QTime, 
    QDateTime, 
    QPoint, 
    QPointF,
)


# Initialize Qt resources from file resources.py
from .resources import *
from .geodb_dialog import GeodbIODialog
import os.path
import json
from .sync import save_model

class GeodbIO:
    """QGIS Plugin Implementation."""

    local = False
    if local is False:
        check_url       = "https://geodb.io/api/check-token/"
        login_url       = "https://geodb.io/api/api-token-auth/"
        logout_url      = "https://geodb.io/api/api-logout/"
        project_url     = "https://geodb.io/api/api-projects/"
        permissions_url = "https://geodb.io/api/api-permissions/"
        data_url        = "https://geodb.io/api/api-data/"

    else:
        check_url       = "http://127.0.0.1:8000/api/check-token/"
        login_url       = "http://127.0.0.1:8000/api/api-token-auth/"
        logout_url      = "http://127.0.0.1:8000/api/api-logout/"
        project_url     = "http://127.0.0.1:8000/api/api-projects/"
        permissions_url = "http://127.0.0.1:8000/api/api-permissions/"
        data_url        = "http://127.0.0.1:8000/api/api-data/"

    from console import console
    console.show_console()


    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GeodbIO_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Geodb.io Connection')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GeodbIO', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):


        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToDatabaseMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/geodb/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'connect now'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginDatabaseMenu(
                self.tr(u'&Geodb.io Connection'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start  == True:
            self.first_start = False
            self.dlg         = GeodbIODialog(plugin=self, parent=self.iface.mainWindow())


        # Check if there is an api_token saved
        config_data, auth_config = self.get_api_token()
        if config_data:
            self.check_token_expiry(config_data)
        else:
            self.toggle_layout_visibility("loginLayout", True)

        # show or hide the assay merge options
        self.toggle_layout_visibility("assayMergeOptions", False)
        self.toggle_assay_merge_options(clear=True)


        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        self.dlg.exec_()

    def check_token_expiry(self, config_data):
        """Check the token expiry by sending it to the server"""

        headers = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}
        response = self.post_request(self.check_url, headers)

        if response: #.status_code == 200:
            expiry = response.get("remaining_time")
            self.dlg.login_status.setText(f"Session expires in: {expiry}")
            #hide loginLayout
            self.toggle_layout_visibility("loginLayout", False)

            # set the companyBox dropdown list to the companies
            self.dlg.companyBox.clear()
            companies   = [company['name'] for company in response['companies']]
            company_box = self.dlg.companyBox
            company_box.addItems(companies)
            #set the selected company and project in the boxes
            self.get_selected_co_proj(set=True)
        else:
            #show loginLayout
            self.toggle_layout_visibility("loginLayout", True)
            email    = config_data['email']
            password = config_data['password']
            self.dlg.email_input.setText(email)
            self.dlg.password_input.setText(password)

    def logout(self):
        """Handle the logout process"""
        config_data, auth_config = self.get_api_token()
        headers = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}
        response = self.post_request(self.logout_url, headers)

        if response: #.status_code == 204: 
            # show the login layout
            self.toggle_layout_visibility("loginLayout", True)
            email    = config_data['email']
            password = config_data['password']
            self.dlg.email_input.setText(email)
            self.dlg.password_input.setText(password)
            self.dlg.login_status.setText(f"Not logged in")
            QMessageBox.warning(self.dlg, 'Logged Out', 'You have successfully logged out.')
            # clear out the company and project dropdowns
            # we will keep the company and projects save in the file though
            self.dlg.companyBox.clear()
            self.dlg.projectBox.clear()
            self.dlg.companyBox.setStyleSheet("background-color: #ffffff;  color: #000000;")
            self.dlg.projectBox.setStyleSheet("background-color: #ffffff;  color: #000000;")
            self.dlg.access_level.setText(f"My Access Level: n/a")

        else:
            print(response)
            QMessageBox.critical(self.dlg, 'Logout Failed', 'Could not log out. Please try again.')

    def login(self, email, password):
        """Handle the login process"""

        # response = requests.post(self.login_url, data={'username': email, 'password': password})
        headers   = {"Content-Type": "application/json"}
        json_data = json.dumps({"username": email, "password": password})
        response = self.post_request(self.login_url, headers, json_data)

        if response:

            token     = response['token']
            expiry    = response['expiry']
            companies = response['companies']

            config_data, existing_config = self.get_api_token()
            authManager = QgsApplication.authManager()

            if existing_config and existing_config.isValid():
                existing_config.setConfig("username", email)
                existing_config.setConfig("password", password)
                existing_config.setConfig("realm", token)  
                authManager.updateAuthenticationConfig(existing_config)
            else:
                config = QgsAuthMethodConfig()
                config.setId(authManager.uniqueConfigId())

                config.setName("geodb_token")
                config.setMethod("Basic")
                config.setUri(self.login_url)
                config.setConfig("username", email)
                config.setConfig("password", password)
                config.setConfig("realm", token)       
                authManager.storeAuthenticationConfig(config)
        
            # populate the company dropdown
            self.dlg.companyBox.clear()
            companies   = [company['name'] for company in companies]
            company_box = self.dlg.companyBox
            company_box.addItems(companies)
            self.get_selected_co_proj(set=True)

            self.dlg.login_status.setText(f"Logged in until: {expiry}")
            # hide login box
            self.toggle_layout_visibility("loginLayout", False)
            # display success message
            QMessageBox.information(self.dlg, 'Login Successful', 'You have successfully logged in.')
        else:
            # Login failed, show an error message
            self.dlg.login_status.setText(f"Not logged in")
            # show login box
            self.toggle_layout_visibility("loginLayout", True)
            # display error message
            QMessageBox.critical(self.dlg, 'Login Failed', 'Invalid email or password. Please try again.')

    def toggle_layout_visibility(self, layout_name, visible):
        layout = self.dlg.findChild(QGridLayout, layout_name)
        if layout:
            for i in range(layout.count()):
                item = layout.itemAt(i)
                if item.widget():
                    item.widget().setVisible(visible)

            #toggle the login button
            self.dlg.logout_submit.setVisible(not visible)

    def set_project(self):
        """
        all we have to do with this one is simple save the project name to the q_project file
        """
        project_name = self.dlg.projectBox.currentText()
        # Get the current project instance
        q_project = QgsProject.instance()
        q_project.writeEntry("geodb_vars", "selected_project", project_name)
        q_project.write()
        # set the the color green once the project is set
        self.dlg.projectBox.setStyleSheet("background-color: #007600;  color: #ffffff;")
        return


    def set_company(self):
        """
        1) save the company to the project file
        2) send the company to the server and get the list of projects
        3) populate the project dropdown

        # to retrieve this value later:
        # value = project.readEntry("geodb_vars", "selected_company")[0]
        """
        company_name = self.dlg.companyBox.currentText()
        # Get the current project instance
        q_project = QgsProject.instance()
        q_project.writeEntry("geodb_vars", "selected_company", company_name)
        q_project.write()

        config_data, auth_config = self.get_api_token()
        headers   = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}
        json_data = json.dumps({"company": company_name})

        response = self.post_request(self.project_url, headers, json_data)

        self.dlg.projectBox.clear()
        projects    = [project['name'] for project in response['projects']]
        project_box = self.dlg.projectBox
        project_box.addItems(projects)

        # set the the color green
        self.dlg.companyBox.setStyleSheet("background-color: #007600;  color: #ffffff;")
        # clear the project box color
        self.dlg.projectBox.setStyleSheet("background-color: #ffffff;  color: #000000;")

        return

    def get_api_token(self):
        """
        Retrieve the saved API token
        used by login, logout, check_token_expiry
        """
        authManager = QgsApplication.authManager()
        auth_configs = authManager.availableAuthMethodConfigs()

        for auth_id, auth_config in auth_configs.items():
            if auth_config.name() == 'geodb_token':
                # get the application's authenticaion manager
                auth_mgr = QgsApplication.authManager()
                # create an empty authmethodconfig object
                auth_cfg = QgsAuthMethodConfig()
                # load config from manager to the new config instance and decrypt sensitive data
                auth_mgr.loadAuthenticationConfig(auth_config.id(), auth_cfg, True)
                # get the configuration information (including username and password)
                auth_info = auth_cfg.configMap()
                data = {
                    'token'    : auth_info.get("realm"),
                    'email'    : auth_info.get("username"),
                    'password' : auth_info.get("password"),
                }
                return data, auth_config
        return None, None
    
    def get_selected_co_proj(self, set=False):
        """
        Retrieve the selected company and project. This is run with both login and token check, so this is where
        we need to ensure that the user has access to both company and project, and get their level of permission. 
        This also sets all of the other plugin variables. 

        The company and project are set with the get_company and get_project functions respectively, so we don't actually query 
        them from the server here. This function sends the company and project to the server to get the user's permissions.

        We also retrieve a list of available models to populate the model dropdowns in pull and push
        """
        q_project = QgsProject.instance()
        try:
            company = q_project.readEntry("geodb_vars", "selected_company")[0]
        except:
            company = None
        try:
            project = q_project.readEntry("geodb_vars", "selected_project")[0]
        except:
            project = None
        try:
            gpkg_file = q_project.readEntry("geodb_vars", "gpkg_filepath")[0]
            self.dlg.gpkg_filepath.setText(gpkg_file)
        except:
            gpkg_file = None
        try:
            model = q_project.readEntry("geodb_vars", "model")[0]
        except:
            model = None
        try:
            include_merged = q_project.readEntry("geodb_vars", "include_merged")[0]
            if include_merged == 'true':
                include_merged = True
            else:
                include_merged = False
            self.dlg.include_mergeCheckBox.setChecked(include_merged)
        except:
            include_merged = None
        try:
            convert_bdl = q_project.readEntry("geodb_vars", "convert_bdl")[0]
            if convert_bdl == 'true':
                convert_bdl = True
            else:
                convert_bdl = False
            self.dlg.convert_bdlCheckBox.setChecked(convert_bdl)
        except:
            convert_bdl = None
        try:
            how_to_merge = q_project.readEntry("geodb_vars", "how_to_merge")[0]
        except:
            how_to_merge = None
        try:
            bdl_ratio = float(q_project.readEntry("geodb_vars", "bdl_ratio")[0])
            self.dlg.bdl_ratioBox.setValue(bdl_ratio)
        except:
            bdl_ratio = None
        try:
            merge_units = q_project.readEntry("geodb_vars", "merge_units")[0]
        except:
            merge_units = None
        try:
            els_list    = q_project.readEntry("geodb_vars", "els_list")[0]
        except:
            els_list = None
        try:
            pull_company = q_project.readEntry("geodb_vars", "include_company_land")[0]
            if pull_company == 'true':
                self.dlg.pullCompanyLand.setChecked(True)
            else:
                self.dlg.pullCompanyLand.setChecked(False)
        except:
            pull_company = None

        
        if set is True:

            # retrieve user permissions
            config_data, auth_config = self.get_api_token()
            headers   = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}
            json_data = json.dumps({'company': company, 'project': project})
            response  = self.post_request(self.permissions_url, headers, json_data)

            try:
                self.dlg.access_level.setText(f"My Access Level: {response['status']}")
            except:
                print(response)

            # populate the model dropdowns
            try:
                self.dlg.pull_modelBox.clear()
                self.dlg.pull_modelBox.addItems(response['available_models'])
                if model:
                    index = self.dlg.pull_modelBox.findText(model)
                    if index != -1:
                        self.dlg.pull_modelBox.setCurrentIndex(index)

                self.dlg.push_modelBox.clear()
                self.dlg.push_modelBox.addItems(response['available_push_models'])
                if model:
                    index = self.dlg.push_modelBox.findText(model)
                    if index != -1:
                        self.dlg.push_modelBox.setCurrentIndex(index)
                
                self.dlg.merge_unitBox.clear()
                self.dlg.merge_unitBox.addItems(response['merge_unit_options'])
                if merge_units:
                    index = self.dlg.merge_unitBox.findText(merge_units)
                    if index != -1:
                        self.dlg.merge_unitBox.setCurrentIndex(index)
                
                self.dlg.assay_mergeBox.clear()
                self.dlg.assay_mergeBox.addItems(response['assay_merge_options'])
                if how_to_merge:
                    index = self.dlg.assay_mergeBox.findText(how_to_merge)
                    if index != -1:
                        self.dlg.assay_mergeBox.setCurrentIndex(index)

                self.dlg.only_elsList.clear()
                self.dlg.only_elsList.setText(els_list)
                
                #color the pull button
                self.dlg.model_pull.setStyleSheet("background-color: #0554ff;  color: #000000;")
            except:
                print(response)

            if company:
                index = self.dlg.companyBox.findText(company)
                if index != -1:
                    self.dlg.companyBox.setCurrentIndex(index)
                    self.dlg.companyBox.setStyleSheet("background-color: #007600;  color: #ffffff;")

            if project:
                # we need to populate the project dropdown first, so we can set the project
                self.set_company()
                index = self.dlg.projectBox.findText(project)
                if index != -1:
                    self.dlg.projectBox.setCurrentIndex(index)
                    self.dlg.projectBox.setStyleSheet("background-color: #007600;  color: #ffffff;")
        
        return company, project
    
    
    def update_progress(self, bytes_received, bytes_total):
        if bytes_total > 0:
            progress_value = (bytes_received * 100) / bytes_total
            self.progress.setValue(int(progress_value))


    def post_request(self, url, headers, data=None, show_progress=False, push=False):
        """handles all of our network calls to the api"""
        manager = QgsNetworkAccessManager.instance()
        request = QNetworkRequest(QUrl(url))
        for key, value in headers.items():
            request.setRawHeader(key.encode(), value.encode())
        
        if data is None:
            data = QByteArray()
        else:
            data = data.encode()

        if show_progress is True:
            # Create progress dialog
            self.progress = QProgressDialog("Downloading data...", "Cancel", 0, 100, self.dlg)
            self.progress.setWindowTitle('Download Progress')
            self.progress.setWindowModality(Qt.WindowModal)
            self.progress.show()

        reply = manager.post(request, data)
        if show_progress is True:
            reply.downloadProgress.connect(self.update_progress)

        # Wait for the request to complete
        loop = QEventLoop()
        reply.finished.connect(loop.quit)
        loop.exec_()

        status_code = reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)
        print('loop finished, status code:', status_code)

        if (push is True) and (status_code == 200):
            # if we successfully pushed the data, we need to update the natural key for the model entries
            print('Pushed data successfully, running data pull to update the model')
            self.data_pull()


        if status_code == 204:
            return {"status": "success", "message": "No Content"}
        
        if reply.error() == QNetworkReply.NoError:
            response_data = str(reply.readAll(), 'utf-8')
            json_data = json.loads(response_data)
            if status_code in [400, 401, 403]:
                # the check_url will return a 401 if the user is not logged in. 
                if url == self.check_url:
                    return response_data
                error_message = json_data.get('error')
                QMessageBox.critical(self.dlg, 'Error', f'{error_message}')
                return 'error'
            return json_data

        else:
            self.dlg.login_status.setText(f"Error: {reply.errorString()}")
            return None
        


    def data_pull(self, model=None):
        file_path = self.dlg.gpkg_filepath.text()
        if not os.path.exists(file_path):
            QMessageBox.critical(self.dlg, 'File Not Found', 'The selected file does not exist.')
            return
        # get model and options from user
        model                = self.dlg.pull_modelBox.currentText()
        include_merged       = self.dlg.include_mergeCheckBox.isChecked()
        include_company_land = self.dlg.pullCompanyLand.isChecked()
        convert_bdl          = self.dlg.convert_bdlCheckBox.isChecked()
        how_to_merge         = self.dlg.assay_mergeBox.currentText()
        bdl_ratio            = self.dlg.bdl_ratioBox.value()
        merge_units          = self.dlg.merge_unitBox.currentText()
        els_list             = self.dlg.only_elsList.text()

        im, cb, cl = 'false', 'false', 'false'
        # when pushed we want to save the state of all the buttons to the qgis project file
        q_project = QgsProject.instance()
        q_project.writeEntry("geodb_vars", "model", model)
        if include_company_land is True:
            cl = 'true'
        q_project.writeEntry("geodb_vars", "include_company_land", cl)
        if include_merged is True:
            im = 'true'
        q_project.writeEntry("geodb_vars", "include_merged", im)
        if convert_bdl is True:
            cb = 'true'
        q_project.writeEntry("geodb_vars", "convert_bdl", cb)
        q_project.writeEntry("geodb_vars", "how_to_merge", how_to_merge)
        q_project.writeEntry("geodb_vars", "bdl_ratio", str(bdl_ratio))
        q_project.writeEntry("geodb_vars", "merge_units", merge_units)
        q_project.writeEntry("geodb_vars", "els_list", els_list)
        q_project.write()


        company, project         = self.get_selected_co_proj()
        config_data, auth_config = self.get_api_token()
        headers                  = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}
        json_data                = json.dumps({'company'              : company, 
                                               'project'              : project, 
                                               'model'                : model, 
                                               'mode'                 : 'pull', 
                                               'include_merged'       : include_merged,
                                               'include_company_land' : include_company_land,
                                               'convert_bdl'          : convert_bdl,
                                               'how_to_merge'         : how_to_merge,
                                               'bdl_ratio'            : bdl_ratio,
                                               'merge_units'          : merge_units,
                                               'els_list'             : els_list,})
        
        data = self.post_request(self.data_url, headers, json_data, show_progress=True)
        if data == 'error':
            return
        if model in ['PointSample' , 'DrillSample', 'Assay', 'Method', 'Standard', 'QCSample']:
            # Because we aren't using the drf serializer from the server for this model, we need to convert the data. 
            # For whatever reason, it comes over as a string, so we need to convert it to a dictionary despite the fact
            #that we have already de-serialized it once in the post_request function.
            data['serialized_data']       = json.loads(data['serialized_data'])
        self.dlg.pull_message.clear()
        if data is None:
            QMessageBox.critical(self.dlg, 'NO DATA', 'No model entries were returned.')
        else:
            save_model(dlg=self.dlg, data=data, model=model)

    def get_push_changes(self):
        """
        The algorithm for this function is as follows:
        1) Get the model from the push_modelBox
        2) send the model to the server to get server data, same as data pull
        3) let the user review the changes
        4) send the changes to the server
        """
        file_path = self.dlg.gpkg_filepath.text()
        if not os.path.exists(file_path):
            QMessageBox.critical(self.dlg, 'File Not Found', 'The selected file does not exist.')
            return
        model                    = self.dlg.push_modelBox.currentText()
        company, project         = self.get_selected_co_proj()
        config_data, auth_config = self.get_api_token()
        headers                  = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}
        json_data                = json.dumps({'company'        : company, 
                                               'project'        : project, 
                                               'model'          : model, 
                                            #    'mode'           : 'push', 
                                               })
        data = self.post_request(self.data_url, headers, json_data, show_progress=True)
        if data == 'error':
            return
        self.dlg.push_message.clear()
        save_model(dlg=self.dlg, data=data, model=model, dir='push')

        q_project = QgsProject.instance()
        q_project.writeEntry("geodb_vars", "model", model)
        self.dlg.model_push_doDelete.setEnabled(True)
        self.dlg.model_push_noDelete.setEnabled(True)
        self.dlg.model_push_getChanges.setStyleSheet("background-color: #007600;  color: #ffffff;")
        self.dlg.model_push_doDelete.setStyleSheet("background-color: #f11f1b;")
        self.dlg.model_push_noDelete.setStyleSheet("background-color: #f1bc1b;")
        

    def data_push(self, do_delete=False):
        file_path = self.dlg.gpkg_filepath.text()
        if not os.path.exists(file_path):
            QMessageBox.critical(self.dlg, 'File Not Found', 'The selected file does not exist.')
            return

        model                    = self.dlg.push_modelBox.currentText()
        company, project         = self.get_selected_co_proj()
        config_data, auth_config = self.get_api_token()
        headers                  = {"Authorization": f"Token {config_data['token']}", "Content-Type": "application/json"}

        # Get the layer by name
        model = 'LandHolding'
        layers = QgsProject.instance().mapLayersByName(model)
        if layers:
            layer = layers[0]  # Get the first layer that matches the name
            # Get all features from the layer
            features = layer.getFeatures()
            # Get srid of layer
            crs = layer.crs()
            # print(f'Layer CRS: {crs.authid()}')
            srid_numeric = int(crs.authid().split(':')[1])
            # print(f'Layer SRID: {srid_numeric}')
            data = {}
            for feature in features:
                # Get attributes as a dictionary and make them serializable
                attr_dict = {
                    field.name(): self.make_serializable(feature.attribute(field.name()))
                    for field in layer.fields() if not any(word in field.name().lower() for word in ['image', 'doc', 'serial_link'])
                }

                try:
                    attr_dict['geometry'] = feature.geometry().asWkt()
                except:
                    attr_dict['geometry'] = None
                try:
                    data[feature.id()] = attr_dict
                except:
                    pass
            
            # print(f'Feature data: {json.dumps(data)}')
            json_data = json.dumps({'company'        : company, 
                                    'project'        : project, 
                                    'model'          : model, 
                                    'mode'           : 'push', 
                                    'do_delete'      : do_delete,
                                    'data'           : json.dumps(data),
                                    'srid'           : srid_numeric,
                                    })
                
            data = self.post_request(self.data_url, headers, json_data, show_progress=True, push=True)
        else:
            QMessageBox.critical(self.dlg, 'Error', 'Layer not found')
            return













    def gpkg_browse(self):
        file_dialog = QFileDialog()
        file_dialog.setFileMode(QFileDialog.AnyFile)
        file_dialog.setNameFilter("GeoPackage files (*.gpkg)")
        file_dialog.setDefaultSuffix("gpkg")
        if file_dialog.exec_():
            selected_files = file_dialog.selectedFiles()
            if selected_files:
                file_path = selected_files[0]
                self.dlg.gpkg_filepath.setText(file_path)
                # Save the file path to the project
                q_project = QgsProject.instance()
                q_project.writeEntry("geodb_vars", "gpkg_filepath", file_path)
                q_project.write()



    def gpkg_checkfile(self):
        gpkg_path = self.dlg.gpkg_filepath.text()

        if not os.path.exists(gpkg_path):
            if gpkg_path == '':
                QMessageBox.information(self.dlg, 'File Required', f'Please select or create a file using the dialog above.')
                return
            

            empty_layer = QgsVectorLayer("None", "empty_layer", "memory")
            # Save the empty layer to the GeoPackage
            options = QgsVectorFileWriter.SaveVectorOptions()
            options.driverName = "GPKG"
            options.fileEncoding = "UTF-8"
            options.layerName = "metadata_layer"  # Name of the metadata table

            QgsVectorFileWriter.writeAsVectorFormatV3(empty_layer, gpkg_path, QgsProject.instance().transformContext(), options)

            # Add a metadata table to store the identifier
            metadata_layer = QgsVectorLayer("None", "metadata", "memory")
            metadata_provider = metadata_layer.dataProvider()

            # Add fields for the metadata table
            metadata_provider.addAttributes([
                QgsField("key", QMetaType.QString),
                QgsField("value", QMetaType.QString),
            ])
            metadata_layer.updateFields()

            # Add the custom identifier as a feature
            feature = QgsVectorLayerUtils.createFeature(metadata_layer)
            feature.setAttributes(["created_by", "my_plugin_identifier"])
            metadata_provider.addFeatures([feature])

            # Save the metadata layer to the GeoPackage
            options.layerName = "metadata"
            QgsVectorFileWriter.writeAsVectorFormatV3(metadata_layer, gpkg_path, QgsProject.instance().transformContext(), options)

            layer = QgsVectorLayer(f"{gpkg_path}|layername=metadata", "Metadata Layer", "ogr")

            if layer.isValid():
                # Add the layer to the project
                QgsProject.instance().addMapLayer(layer)
                print("Layer loaded successfully")
            else:
                print("Failed to load the layer")

            QMessageBox.information(self.dlg, 'File Created', f'GeoPackage created at: {gpkg_path}')
            # print(f"GeoPackage created at: {gpkg_path}")
        else:
            QMessageBox.information(self.dlg, 'File Found', 'The selected file exists.')

    def toggle_push_options(self):
        push_text = self.dlg.push_modelBox.currentText()
        if push_text == 'LandHolding':
            self.dlg.pushLandHoldingNote.setVisible(True)
        else:
            self.dlg.pushLandHoldingNote.setVisible(False)


    def toggle_assay_merge_options(self, clear=False):
        text = self.dlg.pull_modelBox.currentText()
        
        if clear is True:
            self.toggle_layout_visibility("assayMergeOptions", False)
        elif text in ['PointSample', 'DrillSample']:
            self.toggle_layout_visibility("assayMergeOptions", True)
        else:
            self.toggle_layout_visibility("assayMergeOptions", False)

        if text == 'LandHolding':
            self.dlg.pullCompanyLand.setVisible(True)
        else:
            self.dlg.pullCompanyLand.setVisible(False)
            
        self.dlg.model_push_doDelete.setEnabled(False)
        self.dlg.model_push_noDelete.setEnabled(False)
        self.dlg.model_push_getChanges.setStyleSheet("background-color: #efefef;  color: #000000;")
        self.dlg.model_push_doDelete.setStyleSheet("background-color: #efefef;")
        self.dlg.model_push_noDelete.setStyleSheet("background-color: #efefef;")


    def make_serializable(self, value):
        """Convert non-JSON-serializable objects to a serializable format."""
        if value is None:
            return None  # JSON supports null
        elif isinstance(value, QVariant):
            # If QVariant is encountered, attempt to unwrap it safely
            # Since toPyObject() isn’t available, return the value as-is or None if invalid
            return None if value.isNull() else str(value)  # Fallback to string representation
        elif isinstance(value, QDate):
            return value.toString("yyyy-MM-dd")  # e.g., "2025-02-22"
        elif isinstance(value, QTime):
            return value.toString("HH:mm:ss")  # e.g., "14:30:00"
        elif isinstance(value, QDateTime):
            return value.toString("yyyy-MM-ddTHH:mm:ss")  # e.g., "2025-02-22T14:30:00"
        elif isinstance(value, (QPoint, QPointF)):
            return {'x': value.x(), 'y': value.y()}  # Convert to dict with x, y coordinates
        elif isinstance(value, bytes):
            return value.decode('utf-8', errors='replace')  # Decode bytes to string
        elif isinstance(value, (list, tuple)):
            return [self.make_serializable(item) for item in value]  # Recursively handle lists/tuples
        elif isinstance(value, dict):
            return {key: self.make_serializable(val) for key, val in value.items()}  # Recursively handle dicts
        return value  # Return as-is if no special handling is needed